<?php
/**
 * Plugin Name: Google Indexer Software Automation
 * Description: Manage API credentials, auto-index settings, manual indexing, account info, and history for Google Indexer. Also includes Googlebot visit tracking.
 * Version: 1.0.1
 * Author: Indexer Software
 * Author URI: https://indexersoftware.com 
 * Text Domain: google-indexer-automation
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Eklenti etkinleştirildiğinde çalışarak gerekli tabloları oluşturur veya günceller.
 */
function gia_activate_plugin() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();
    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

    // Indexer History Tablosu
    $table_name_indexer = $wpdb->prefix . 'gia_history';
    $sql_indexer = "CREATE TABLE $table_name_indexer (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        url text NOT NULL,
        indexed_at datetime NOT NULL,
        PRIMARY KEY  (id)
    ) $charset_collate;";
    dbDelta( $sql_indexer );

    // Googlebot History Tablosu
    $table_name_googlebot = $wpdb->prefix . 'gia_googlebot_history';
    $sql_googlebot = "CREATE TABLE $table_name_googlebot (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        url text NOT NULL,
        visited_at datetime NOT NULL,
        ip_address varchar(100) NOT NULL,
        user_agent text NOT NULL,
        PRIMARY KEY  (id)
    ) $charset_collate;";
    dbDelta( $sql_googlebot );
}
register_activation_hook( __FILE__, 'gia_activate_plugin' );


class Google_Indexer_Automation {
    private static $instance = null;
    public static function instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
            self::$instance->init_hooks();
        }
        return self::$instance;
    }

    private function init_hooks() {
        // Admin menu and settings
        add_action( 'admin_menu', [ $this, 'register_admin_menu' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );

        // Admin Notices
        add_action( 'admin_notices', [ $this, 'show_send_to_index_notice' ] );

        // Manual indexing and history clear
        add_action( 'admin_post_gia_manual_index', [ $this, 'handle_manual_index' ] );
        add_action( 'admin_post_gia_clear_history', [ $this, 'handle_clear_history' ] );
        
        // Row action "Send to Index"
        add_action( 'admin_post_gia_send_index', [ $this, 'handle_send_index' ] );
        add_filter( 'post_row_actions', [ $this, 'add_send_action_link' ], 10, 2 );
        add_filter( 'page_row_actions', [ $this, 'add_send_action_link' ], 10, 2 );

        // Auto-index hooks (triggers background job)
        if ( get_option( 'gia_auto_post_publish' ) ) {
            add_action( 'publish_post', [ $this, 'schedule_rate_limited_send' ], 10, 1 );
        }
        if ( get_option( 'gia_auto_page_publish' ) ) {
            add_action( 'publish_page', [ $this, 'schedule_rate_limited_send' ], 10, 1 );
        }
        if ( get_option( 'gia_auto_post_update' ) || get_option( 'gia_auto_page_update' ) ) {
            add_action( 'post_updated', [ $this, 'handle_post_update' ], 10, 3 );
        }
        
        // Arka plan görevini yürüten kanca
        add_action( 'gia_send_url_to_api_background', [ $this, 'send_to_indexer_api' ], 10, 1 );

        // Googlebot takip kancaları
        add_action( 'init', [ $this, 'handle_googlebot_visit' ], 0 );
        add_action( 'admin_post_gia_clear_googlebot_history', [ $this, 'handle_clear_googlebot_history' ] );
    }
    
    public function show_send_to_index_notice() {
        if ( ! isset( $_GET['gia_sent_to_index'] ) ) {
            return;
        }

        if ( '1' === $_GET['gia_sent_to_index'] ) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Link has been successfully sent to the indexer.', 'google-indexer-automation' ) . '</p></div>';
        } elseif ( 'error_credentials' === $_GET['gia_sent_to_index'] ) {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__( 'Could not send link. Please enter and save your credentials on the Login page first.', 'google-indexer-automation' ) . '</p></div>';
        }
    }

    public function register_admin_menu() {
        add_menu_page(
            __( 'Indexer', 'google-indexer-automation' ),
            __( 'Indexer', 'google-indexer-automation' ),
            'manage_options',
            'gia-login',
            [ $this, 'login_page_callback' ],
            'dashicons-admin-network',
            80
        );
        add_submenu_page( 'gia-login', __( 'Login', 'google-indexer-automation' ), __( 'Login', 'google-indexer-automation' ), 'manage_options', 'gia-login', [ $this, 'login_page_callback' ] );
        add_submenu_page( 'gia-login', __( 'Auto Index', 'google-indexer-automation' ), __( 'Auto Index', 'google-indexer-automation' ), 'manage_options', 'gia-auto-index', [ $this, 'auto_index_page_callback' ] );
        add_submenu_page( 'gia-login', __( 'Manual Index', 'google-indexer-automation' ), __( 'Manual Index', 'google-indexer-automation' ), 'manage_options', 'gia-manual-index', [ $this, 'manual_index_page_callback' ] );
        add_submenu_page( 'gia-login', __( 'Account', 'google-indexer-automation' ), __( 'Account', 'google-indexer-automation' ), 'manage_options', 'gia-account', [ $this, 'account_page_callback' ] );
        add_submenu_page( 'gia-login', __( 'History', 'google-indexer-automation' ), __( 'History', 'google-indexer-automation' ), 'manage_options', 'gia-history', [ $this, 'history_page_callback' ] );
        add_submenu_page( 'gia-login', __( 'Googlebot', 'google-indexer-automation' ), __( 'Googlebot', 'google-indexer-automation' ), 'manage_options', 'gia-googlebot', [ $this, 'googlebot_page_callback' ] );
    }

    public function register_settings() {
        // Login
        register_setting( 'gia_login_group', 'gia_email', [ 'sanitize_callback' => 'sanitize_email' ] );
        register_setting( 'gia_login_group', 'gia_verification_code', [ 'sanitize_callback' => 'sanitize_text_field' ] );
        add_settings_section( 'gia_login_section', '', '__return_false', 'gia-login' );
        add_settings_field( 'gia_field_email', __( 'Email Address', 'google-indexer-automation' ), [ $this, 'field_email_cb' ], 'gia-login', 'gia_login_section' );
        add_settings_field( 'gia_field_code',  __( 'Verification Code', 'google-indexer-automation' ), [ $this, 'field_code_cb' ],  'gia-login', 'gia_login_section' );

        // Auto Index Settings
        register_setting( 'gia_auto_group', 'gia_auto_post_publish' );
        register_setting( 'gia_auto_group', 'gia_auto_post_update' );
        register_setting( 'gia_auto_group', 'gia_auto_page_publish' );
        register_setting( 'gia_auto_group', 'gia_auto_page_update' );
        add_settings_section( 'gia_auto_section', __( 'Choose when to automatically send links to the indexer', 'google-indexer-automation' ), '__return_false', 'gia-auto-index' );
        
        $opts = [
            'gia_auto_post_publish' => __( 'When a new Post is published', 'google-indexer-automation' ),
            'gia_auto_post_update'  => __( 'When any Post is updated/saved', 'google-indexer-automation' ),
            'gia_auto_page_publish' => __( 'When a new Page is published', 'google-indexer-automation' ),
            'gia_auto_page_update'  => __( 'When any Page is updated/saved', 'google-indexer-automation' ),
        ];

        foreach ( $opts as $key => $label ) {
            add_settings_field( $key, $label, [ $this, 'field_checkbox_cb' ], 'gia-auto-index', 'gia_auto_section', [ 'option_name' => $key ] );
        }
    }

    public function field_email_cb() {
        printf( '<input type="email" name="gia_email" value="%s" class="regular-text" required />', esc_attr( get_option( 'gia_email', '' ) ) );
    }

    public function field_code_cb() {
        printf( '<input type="text" name="gia_verification_code" value="%s" class="regular-text" required />', esc_attr( get_option( 'gia_verification_code', '' ) ) );
    }

    public function field_checkbox_cb( $args ) {
        $opt     = $args['option_name'];
        $checked = get_option( $opt ) ? 'checked' : '';
        printf( '<label><input type="checkbox" name="%s" value="1" %s /></label>', esc_attr( $opt ), $checked );
    }

    public function login_page_callback() {
		if ( isset( $_GET['settings-updated'] ) ) {
			$this->test_api_connection();
		}
		settings_errors( 'gia_login_group' );
		echo '<div class="wrap"><h1>' . esc_html__( 'Login to Indexer API', 'google-indexer-automation' ) . '</h1>';
		echo '<p>' . esc_html__( 'You can find your Verification Code (also known as Authcode) within our application, or in the email that was sent to you upon registration.', 'google-indexer-automation' ) . '</p>';

		echo '<form method="post" action="options.php">';
		settings_fields( 'gia_login_group' );
		do_settings_sections( 'gia-login' );
		submit_button( __( 'Save Credentials', 'google-indexer-automation' ) );
		echo '</form></div>';
	}


    public function auto_index_page_callback() {
        echo '<div class="wrap"><h1>' . esc_html__( 'Auto Index Settings', 'google-indexer-automation' ) . '</h1>';
        
        if ( isset( $_GET['settings-updated'] ) && $_GET['settings-updated'] ) {
            add_settings_error( 'gia_auto_group', 'settings_updated', __( 'Settings saved successfully.', 'google-indexer-automation' ), 'updated' );
        }
        settings_errors( 'gia_auto_group' );
        
        echo '<form method="post" action="options.php">';
        settings_fields( 'gia_auto_group' );
        do_settings_sections( 'gia-auto-index' );
        submit_button( __( 'Save Auto Index Settings', 'google-indexer-automation' ) );
        echo '</form></div>';
    }

    public function manual_index_page_callback() {
        if ( isset( $_GET['gia_manual_success'] ) && '1' === $_GET['gia_manual_success'] ) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Links sent successfully and saved to history.', 'google-indexer-automation' ) . '</p></div>';
        }
        if ( isset( $_GET['gia_manual_error'] ) && 'credentials' === $_GET['gia_manual_error'] ) {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__( 'Please enter and save your Email and Verification Code on the Login page before sending links.', 'google-indexer-automation' ) . '</p></div>';
        }

        echo '<div class="wrap"><h1>' . esc_html__( 'Manual Index', 'google-indexer-automation' ) . '</h1>';
        echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '">';
        wp_nonce_field( 'gia_manual_index', 'gia_manual_index_nonce' );
        echo '<input type="hidden" name="action" value="gia_manual_index" />';
        echo '<textarea name="gia_manual_links" rows="10" class="large-text" placeholder="Enter one URL per line"></textarea>';
        submit_button( __( 'Send Links', 'google-indexer-automation' ) );
        echo '</form></div>';
    }

    public function history_page_callback() {
        global $wpdb;
        $table = $wpdb->prefix . 'gia_history';
        $rows  = $wpdb->get_results( "SELECT * FROM {$table} ORDER BY indexed_at DESC" );
        
        if ( isset( $_GET['history_cleared'] ) ) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Indexing history has been cleared.', 'google-indexer-automation' ) . '</p></div>';
        }

        echo '<div class="wrap"><h1>' . esc_html__( 'Indexing History', 'google-indexer-automation' ) . '</h1>';
        echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" onsubmit="return confirm(\'' . esc_js( __( 'Are you sure you want to clear all history?', 'google-indexer-automation' ) ) . '\');">';
        wp_nonce_field( 'gia_clear_history', 'gia_clear_history_nonce' );
        echo '<input type="hidden" name="action" value="gia_clear_history" />';
        submit_button( __( 'Clear All History', 'google-indexer-automation' ), 'delete' );
        echo '</form>';

        if ( $rows ) {
            echo '<table class="widefat fixed striped"><thead><tr><th>' . esc_html__( 'URL', 'google-indexer-automation' ) . '</th><th>' . esc_html__( 'Crawled At', 'google-indexer-automation' ) . '</th><th>' . esc_html__( 'Check', 'google-indexer-automation' ) . '</th></tr></thead><tbody>';
            foreach ( $rows as $row ) {
                $url       = esc_html( $row->url );
                $time      = esc_html( $row->indexed_at );
                $check_url = 'https://www.google.com/search?q=' . urlencode( 'site:' . $row->url );
                $check_btn = '<a href="' . esc_url( $check_url ) . '" class="button button-secondary" target="_blank" rel="noopener noreferrer">' . esc_html__( 'Check', 'google-indexer-automation' ) . '</a>';
                
                echo "<tr><td>{$url}</td><td>{$time}</td><td>" . $check_btn . "</td></tr>";
            }
            echo '</tbody></table>';
        } else {
            echo '<p>' . esc_html__( 'No history found.', 'google-indexer-automation' ) . '</p>';
        }
        echo '</div>';
    }

    public function account_page_callback() {
        if ( isset( $_GET['check_balance'] ) ) {
            $bal = $this->get_balance();
            if ( false === $bal ) {
                add_settings_error( 'gia_account', 'balance_error', __( 'Could not retrieve balance.', 'google-indexer-automation' ), 'error' );
            } else {
                add_settings_error( 'gia_account', 'balance_success', sprintf( __( 'Current balance: %s', 'google-indexer-automation' ), esc_html( $bal ) ), 'updated' );
            }
        }
        settings_errors( 'gia_account' );
        echo '<div class="wrap"><h1>' . esc_html__( 'Account Info', 'google-indexer-automation' ) . '</h1><table class="form-table"><tbody>';
        printf( '<tr><th>%s</th><td>%s</td></tr>', esc_html__( 'Email Address', 'google-indexer-automation' ), esc_html( get_option( 'gia_email', '' ) ) );
        printf( '<tr><th>%s</th><td>%s</td></tr>', esc_html__( 'Verification Code', 'google-indexer-automation' ), esc_html( get_option( 'gia_verification_code', '' ) ) );
        printf( '<tr><th>%s</th><td>%s</td></tr>', esc_html__( 'Balance', 'google-indexer-automation' ), esc_html( $this->get_balance() ) );
        echo '</tbody></table><form method="get"><input type="hidden" name="page" value="gia-account" /><button type="submit" name="check_balance" class="button button-secondary">' . esc_html__( 'Check Balance', 'google-indexer-automation' ) . '</button></form></div>';
    }

    private function test_api_connection() {
		$email = get_option('gia_email');
		$code  = get_option('gia_verification_code');

		if ( ! $email || ! $code ) {
			return; // Alanlar boşsa teste devam etme
		}

		// Bakiye kontrolü yaparak kimlik bilgilerini doğrula
		$response = wp_remote_post('https://api.indexersoftware.com/api/getcredit', [
			'headers' => [
				'Authorization' => 'Basic ' . base64_encode("{$email}:{$code}"),
				'Usage'         => 'WP_INDEXER',
				'Content-Type'  => 'application/json',
			],
			'body'    => wp_json_encode(['Email' => $email]),
			'timeout' => 15,
		]);

		if (is_wp_error($response)) {
			// WordPress veya sunucu kaynaklı bağlantı hatası
			$error_message = $response->get_error_message();
			add_settings_error('gia_login_group', 'conn_error', __('Connection failed: ', 'google-indexer-automation') . $error_message, 'error');
		} else {
			$http_code = wp_remote_retrieve_response_code($response);
			$body      = wp_remote_retrieve_body($response);
			$data      = json_decode($body, true);

			if ($http_code === 200 && isset($data['Data'])) {
				// Başarılı giriş
				add_settings_error('gia_login_group', 'conn_success', __('Login successful. API connection is working correctly.', 'google-indexer-automation'), 'success');
			} else {
				// API'den gelen başarısız giriş yanıtı
				$api_error_message = isset($data['Message']) ? $data['Message'] : __('Invalid credentials or unknown API error. Please check your Email and Verification Code.', 'google-indexer-automation');
				add_settings_error('gia_login_group', 'conn_failed', __('Login failed: ', 'google-indexer-automation') . esc_html($api_error_message), 'error');
			}
		}
	}


    private function get_balance() {
        $email = get_option('gia_email');
        $code  = get_option('gia_verification_code');

        $res = wp_remote_post('https://api.indexersoftware.com/api/getcredit', [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode("{$email}:{$code}"),
                'Usage'         => 'WP_INDEXER',
                'Content-Type'  => 'application/json',
            ],
            'body'    => wp_json_encode(['Email' => $email]),
            'timeout' => 10,
        ]);

        if (is_wp_error($res)) {
            return false;
        }

        $body = wp_remote_retrieve_body($res);
        $data = json_decode($body, true);

        if (isset($data['Data'])) {
            return $data['Data'];
        }
        
        return false;
    }

    public function send_to_indexer_api( $url ) {
        $email = get_option( 'gia_email' );
        $code  = get_option( 'gia_verification_code' );

        if ( ! $email || ! $code || ! is_string( $url ) || empty( $url ) ) {
            return;
        }

        $response = wp_remote_post( 'https://api.indexersoftware.com/api/sendLink', [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode( "{$email}:{$code}" ),
                'Content-Type'  => 'application/json',
                'Usage'         => 'WP_INDEXER'
            ],
            'body'    => wp_json_encode( [
                'Email'            => $email,
                'VerificationCode' => $code,
                'Links'            => [ $url ]
            ] ),
            'timeout' => 20,
        ] );

        if ( is_wp_error( $response ) ) {
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'gia_history';
        $wpdb->insert(
            $table_name,
            [
                'url'        => $url,
                'indexed_at' => current_time( 'mysql' ),
            ]
        );
    }

    public function schedule_rate_limited_send( $post_ID ) {
        $url = get_permalink( $post_ID );
        if ( ! $url || ! is_string( $url ) ) return;

        $key       = 'gia_rate_' . md5( $url );
        $last_time = get_transient( $key );
        if ( false !== $last_time ) {
            return;
        }
        
        wp_schedule_single_event( time(), 'gia_send_url_to_api_background', [ 'url' => $url ] );
        
        set_transient( $key, time(), 30 );
    }

    public function handle_post_update( $post_ID, $post_after, $post_before ) {
        if ( $post_before->post_status !== 'publish' || $post_after->post_status !== 'publish' ) {
            return;
        }

        if ( 'post' === $post_after->post_type && get_option( 'gia_auto_post_update' ) ) {
            $this->schedule_rate_limited_send( $post_ID );
        }
        if ( 'page' === $post_after->post_type && get_option( 'gia_auto_page_update' ) ) {
            $this->schedule_rate_limited_send( $post_ID );
        }
    }

    public function handle_manual_index() {
        if ( ! current_user_can( 'manage_options' ) || ! isset( $_POST['gia_manual_index_nonce'] ) || ! wp_verify_nonce( $_POST['gia_manual_index_nonce'], 'gia_manual_index' ) ) {
            wp_die( esc_html__( 'Unauthorized', 'google-indexer-automation' ) );
        }

        $email = get_option( 'gia_email' );
        $code  = get_option( 'gia_verification_code' );
        if ( empty($email) || empty($code) ) {
            wp_safe_redirect( add_query_arg( 'gia_manual_error', 'credentials', wp_get_referer() ) );
            exit;
        }

        if ( empty( $_POST['gia_manual_links'] ) ) {
            wp_safe_redirect( add_query_arg( 'gia_manual_success', 0, wp_get_referer() ) );
            exit;
        }
        $lines = preg_split( '/\r\n|\r|\n/', sanitize_textarea_field( $_POST['gia_manual_links'] ) );
        foreach ( $lines as $url ) {
            $url = trim( $url );
            if ( $url ) {
                if ( ! preg_match( '/^https?:\/\//i', $url ) ) {
                    $url = 'https://' . $url;
                }
                $this->send_to_indexer_api( esc_url_raw( $url ) );
            }
        }
        wp_safe_redirect( add_query_arg( 'gia_manual_success', 1, wp_get_referer() ) );
        exit;
    }

    public function handle_clear_history() {
        if ( ! current_user_can( 'manage_options' ) || ! isset( $_POST['gia_clear_history_nonce'] ) || ! wp_verify_nonce( $_POST['gia_clear_history_nonce'], 'gia_clear_history' ) ) {
            wp_die( esc_html__( 'Unauthorized request', 'google-indexer-automation' ) );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'gia_history';
        $wpdb->query( "TRUNCATE TABLE {$table}" );

        wp_safe_redirect( add_query_arg( 'history_cleared', 1, wp_get_referer() ) );
        exit;
    }

    public function add_send_action_link( $actions, $post ) {
        $nonce = wp_create_nonce( 'gia_send_index_post_' . $post->ID );
        $url   = admin_url( 'admin-post.php?action=gia_send_index&post_id=' . $post->ID . '&_wpnonce=' . $nonce );
        $actions['send_to_index'] = '<a href="' . esc_url( $url ) . '">' . esc_html__( 'Send to Index', 'google-indexer-automation' ) . '</a>';
        return $actions;
    }

    public function handle_send_index() {
        if ( ! current_user_can( 'manage_options' ) ||
             ! isset( $_GET['post_id'], $_GET['_wpnonce'] ) ||
             ! wp_verify_nonce( $_GET['_wpnonce'], 'gia_send_index_post_' . intval( $_GET['post_id'] ) ) ) {
            wp_die( esc_html__( 'Unauthorized request', 'google-indexer-automation' ) );
        }

        $email = get_option( 'gia_email' );
        $code  = get_option( 'gia_verification_code' );
        if ( empty($email) || empty($code) ) {
            $redirect_url = add_query_arg( 'gia_sent_to_index', 'error_credentials', wp_get_referer() );
            wp_safe_redirect( $redirect_url );
            exit;
        }

        $post_id = intval( $_GET['post_id'] );
        $url     = get_permalink( $post_id );
        if ( $url ) {
            $this->send_to_indexer_api( $url );
        }
        
        $redirect_url = add_query_arg( 'gia_sent_to_index', '1', wp_get_referer() );
        wp_safe_redirect( $redirect_url );
        exit;
    }

    // ===================================================================
    // GOOGLEBOT TAKİP FONKSİYONLARI
    // ===================================================================

    public function handle_googlebot_visit() {
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        if (stripos($user_agent, 'Googlebot') === false) {
            return;
        }

        if (wp_doing_ajax() || is_admin()) {
            return;
        }

        $authkey = get_option('gia_verification_code');
        if (empty($authkey)) {
            error_log('GIA Error: Verification code (authkey) is empty.');
            return;
        }

        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $current_url = $protocol . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];

        $data = [
            'authkey'   => $authkey,
            'clientIp'  => $_SERVER['REMOTE_ADDR'] ?? 'UNKNOWN_IP',
            'clientUrl' => $current_url,
            'userAgent' => $user_agent,
        ];

        global $wpdb;
        $table_name = $wpdb->prefix . 'gia_googlebot_history';
        $wpdb->insert(
            $table_name,
            [
                'url'        => $data['clientUrl'],
                'visited_at' => current_time('mysql'),
                'ip_address' => $data['clientIp'],
                'user_agent' => $data['userAgent'],
            ]
        );

        // HATA AYIKLAMA İÇİN DEĞİŞTİRİLDİ
        $args = [
            'body'     => wp_json_encode($data),
            'headers'  => [
                'Content-Type' => 'application/json',
                'Usage'        => 'WP_INDEXER',
            ],
            'timeout'  => 30, // Timeout süresi artırıldı
            'blocking' => true, // API yanıtını beklemek için 'true' yapıldı
        ];

        $response = wp_remote_post("https://api.indexersoftware.com:1955", $args);

        // Hata ayıklama için loglama eklendi
        if ( is_wp_error( $response ) ) {
            error_log( 'GIA Googlebot API Hatası: ' . $response->get_error_message() );
        } else {
            $http_code = wp_remote_retrieve_response_code( $response );
            $body = wp_remote_retrieve_body( $response );
            error_log( "GIA Googlebot API Yanıtı: Kod=$http_code, Gövde=$body" );
        }
        // HATA AYIKLAMA SONU
    }
    
    public function googlebot_page_callback() {
        global $wpdb;
        $table = $wpdb->prefix . 'gia_googlebot_history';
        $rows  = $wpdb->get_results( "SELECT * FROM {$table} ORDER BY visited_at DESC LIMIT 2000" );
        
        if ( isset( $_GET['googlebot_history_cleared'] ) ) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Googlebot visit history has been cleared.', 'google-indexer-automation' ) . '</p></div>';
        }

        echo '<div class="wrap"><h1>' . esc_html__( 'Googlebot Visit History', 'google-indexer-automation' ) . '</h1>';
        echo '<p>' . esc_html__( 'This page shows the last 2000 visits from detected Googlebots. Only actual page visits are recorded, background requests are ignored. If you cannot watch live bots on the "Watch Live Googlebot Visits" screen, ask your hosting company to allow you to communicate with the ip:port "69.10.63.90:1955". Also, to prevent table bloating, Googlebot visits to media files are not reflected in the table.', 'google-indexer-automation' ) . '</p>';
        
        $authkey = get_option('gia_verification_code');
        if ( $authkey ) {
            $live_url = 'https://api.indexersoftware.com:1956/realtimeBotVisitor.html?authkey=' . esc_attr($authkey);
            echo '<p><a href="' . esc_url($live_url) . '" target="_blank" class="button button-primary">' . esc_html__('Watch Live Googlebot Visits', 'google-indexer-automation') . '</a></p>';
        }
        
        echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" onsubmit="return confirm(\'' . esc_js( __( 'Are you sure you want to clear all Googlebot history?', 'google-indexer-automation' ) ) . '\');">';
        wp_nonce_field( 'gia_clear_googlebot_history', 'gia_clear_googlebot_history_nonce' );
        echo '<input type="hidden" name="action" value="gia_clear_googlebot_history" />';
        submit_button( __( 'Clear Googlebot History', 'google-indexer-automation' ), 'delete' );
        echo '</form>';

        if ( $rows ) {
            echo '<table class="widefat fixed striped"><thead><tr>';
            echo '<th>' . esc_html__( 'Visited URL', 'google-indexer-automation' ) . '</th>';
            echo '<th>' . esc_html__( 'Time', 'google-indexer-automation' ) . '</th>';
            echo '<th>' . esc_html__( 'IP Address', 'google-indexer-automation' ) . '</th>';
            echo '<th>' . esc_html__( 'User Agent', 'google-indexer-automation' ) . '</th>';
            echo '</tr></thead><tbody>';
            
            foreach ( $rows as $row ) {
                echo "<tr>";
                echo "<td>" . esc_html( $row->url ) . "</td>";
                echo "<td>" . esc_html( $row->visited_at ) . "</td>";
                echo "<td>" . esc_html( $row->ip_address ) . "</td>";
                echo "<td>" . esc_html( $row->user_agent ) . "</td>";
                echo "</tr>";
            }
            echo '</tbody></table>';
        } else {
            echo '<p>' . esc_html__( 'No Googlebot visits recorded yet.', 'google-indexer-automation' ) . '</p>';
        }
        echo '</div>';
    }

    public function handle_clear_googlebot_history() {
        if ( ! current_user_can( 'manage_options' ) || ! isset( $_POST['gia_clear_googlebot_history_nonce'] ) || ! wp_verify_nonce( $_POST['gia_clear_googlebot_history_nonce'], 'gia_clear_googlebot_history' ) ) {
            wp_die( esc_html__( 'Unauthorized request', 'google-indexer-automation' ) );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'gia_googlebot_history';
        $wpdb->query( "TRUNCATE TABLE {$table}" );

        $redirect_url = add_query_arg( [ 'page' => 'gia-googlebot', 'googlebot_history_cleared' => 1 ], admin_url('admin.php') );
        wp_safe_redirect( $redirect_url );
        exit;
    }
}
add_action( 'plugins_loaded', [ 'Google_Indexer_Automation', 'instance' ] );